/*
 * Copyright (C) Jan 2006 Mellanox Technologies Ltd. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 *  PerlClass.cpp - Perl low-level interface implementation
 *
 *  Version: $Id: PerlClass.cpp 2752 2006-01-19 14:40:17Z mst $
 *
 */
#include <stdio.h>
#include <stdlib.h>

#include <map>
#include <list>
#include <algorithm>

//#if defined(__CYGWIN32__)
#include "compatibility.h"
//#endif


#include <EXTERN.h>

//
// HACK: The win32.h n Perl/lib/CORE defines this macro even if it's already defined.
//
#ifdef __WIN__
#ifdef WIN32_LEAN_AND_MEAN
#undef WIN32_LEAN_AND_MEAN
#endif

#ifdef DEBUG
#undef DEBUG
#endif
#endif

#include <perl.h>
#include "XSUB.h"
// Only bloody idiots may write "#define list" in their header files!
// orenk: in addition to list, Win32 perl defines read, write, eof, and bool. I tend to agree 
// with the above line for these macros too.
#undef list
#undef read
#undef write
#undef eof
#undef bool


#include "PerlClass.h"
#include "Param.h"
#include "ParamList.h"




bool            debugPerl;
PerlInterpreter *perl_interp;

static ParamList       *plist;
static const char*      g_source  = "Perl";
static int             g_line    = -1;

namespace std {}; using namespace std;

////////////////////////////////////////////////////////////////////////
// Get string value from HASH. Use "def" as default
static char *get_value(HV* hv, const char *key, const char *def=0)
{
    SV **psv = hv_fetch(hv, key, strlen(key), 0);
    if (!psv)
    {
        if (def)
            return (char *)def;
        else
            croak("padd: key \"%s\" must be present.", key);
    }

    return SvPV_nolen(*psv);
} // get_value

////////////////////////////////////////////////////////////////////////
// Get integer value from HASH. Use "def" as default
static u_int32_t get_ivalue(HV* hv, const char *key, const char *def=0)
{
    char      *endp;
    u_int32_t rc = strtoul(get_value(hv, key, def), &endp, 0);

    if (*endp)
        croak("padd: key \"%s\" has invalid value (%s). Must be integer.",
              key, get_value(hv, key, def));
    return rc;
} // get_ivalue


static Param* get_param(const char * sym) {
    map<string, Param*>::iterator npit = plist->params.find(sym);    // name iterator
    map<string, Param*>::iterator rpit = plist->refparams.find(sym); // refname iterator
    Param* p = NULL;
    
    // Check if parameter exists
    if (npit == plist->params.end() && rpit == plist->refparams.end())
	croak("Symbol (refname) \"%s\" doesn't exist", sym);
    else if (npit != plist->params.end() && rpit != plist->refparams.end()) {
	if (npit->second != rpit->second) {
	    croak("Symbol \"%s\" matches refname and name for 2 different parameters", sym);
	}
	p = npit->second;
    }
    else if (npit != plist->params.end()) {
	p = npit->second;
    }
    else if (rpit != plist->refparams.end()) {
	p = rpit->second;
    }

    return p;
} 


////////////////////////////////////////////////////////////////////////
//     pcalculateGUID($COMPANY_ID,$TYPE,$day,$mon,$yer,$num,$cc)      //
////////////////////////////////////////////////////////////////////////
XS(pcalculateGUID)
{
    dXSARGS;
    if (items != 7)
        croak("Usage: pcalculateGUID(COMPANY_ID,TYPE,day,mon,year,num,cc)");
    SP -= items;
    {
        u_int64_t  COMPANY_ID = SvUV(ST(0));
        u_int64_t  TYPE       = SvUV(ST(1));
        u_int32_t  day        = SvUV(ST(2));
        u_int32_t  mon        = SvUV(ST(3));
        u_int32_t  yer        = SvUV(ST(4));
        u_int32_t  num        = SvUV(ST(5));
        u_int32_t  cc         = SvUV(ST(6));
        char       RETVAL[20];

        u_int64_t  rc = ((((yer*12+mon-1)*31+ day-1) * 1000) + num-1) * 112;
        rc += (cc-1)*8;
        rc |= (COMPANY_ID << 40) | (TYPE << 32);
        sprintf(RETVAL, "0x%016"U64L"x", rc);

        if (debugPerl)
            printf("pcalculateGUID(0x%"U64L"x,%"U64L"d,%d,%d,%d,%d,%d) = %s\n",
                   COMPANY_ID, TYPE, day, mon, yer, num, cc, RETVAL);
        XPUSHs(sv_2mortal(newSVpv(RETVAL, strlen(RETVAL))));
    }
    XSRETURN(1);
} // pcalculateGUID


////////////////////////////////////////////////////////////////////////
//                      pdelete($refname)                             //
////////////////////////////////////////////////////////////////////////
XS(pdelete)
{
    dXSARGS;
    if (items != 1)
        croak("Usage: delete(refname)");
    SP -= items;
    {
        char                          *refname = (char *)SvPV_nolen(ST(0));


	//orenk
	croak("pdelete: function not supported (trying to delete param \"%s\".", refname);

        map<string, Param*>::iterator pit = plist->params.find(refname);

        // Check if parameter exists
        if (pit == plist->params.end())
            croak("pdelete: Parameter with refname \"%s\" doesn't exist", refname);

        // Erase it from global parameters map
        plist->params.erase(pit);

        // Erase it from list in its group
        Param *param = pit->second;
        list<Param*>::iterator pgit = find(param->group->prlist.begin(),
                                           param->group->prlist.end(), param);
        if (pgit == param->group->prlist.end())
            croak("pdelete: Internal error, Group \"%s\" doesn't contain "
                  "parameter \"%s\"", param->group->name.c_str(), refname);
        param->group->prlist.erase(pgit);
        delete param;
    }
    XSRETURN_EMPTY;
} // pdelete



////////////////////////////////////////////////////////////////////////
//                         padd($refname)                             //
////////////////////////////////////////////////////////////////////////
XS(padd)
{
    dXSARGS;

    if (items != 2)
        croak("Usage: padd(refname, attributes_hash_reference)");
    SP -= items;
    {
        char *refname  = (char *)SvPV_nolen(ST(0));
        SV   *hashr = ST(1);

	//orenk:
	croak("padd: function not supported (trying to add param \"%s\".", refname);


        // Check parameters
        if (!SvROK(hashr))
            croak("padd: second parameter must be reference.");
        if (SvTYPE(SvRV(hashr)) != SVt_PVHV)
            croak("padd: second parameter must be reference to hash.");

        HV   *hv = (HV *)SvRV(hashr);
        if (debugPerl)
        {
            SV   *sv;
            char *key;
            I32  len;

            printf("padd(\"%s\",\n", refname);
            while ((sv = hv_iternextsv(hv, &key, &len)) != 0)
                printf("    \"%s\" => \"%s\",\n", key, SvPV_nolen(sv));
            printf("    )\n");
         }

        // Check if parameter already exists
        map<string, Param*>::iterator pit = plist->params.find(refname);
        if (pit != plist->params.end())
            croak("padd: Parameter with refname \"%s\" already exists", refname);

        // +++++
        // +++++FIXME: Need to implement real parameter addition
        // +++++
    }
    XSRETURN_EMPTY;
} // padd

////////////////////////////////////////////////////////////////////////
//               pmessage($msg [, $fatal_flag])                       //
////////////////////////////////////////////////////////////////////////
XS(pmessage)
{
    dXSARGS;
    if (items != 2  &&  items != 1)
	croak("Usage: pmessage(msg [, fatal_flag])");
    SP -= items;
    {
        char   *msg = (char *)SvPV_nolen(ST(0));
        int    fatal = (items == 2) ? SvUV(ST(1)) : 0;
        int    rc = 0;

        if (fatal)
        {
            croak("-E- Perl Error: %s\n", msg);
        }
        else
        {
            printf("-W- Perl Message: %s\n", msg);
        }

        if (debugPerl)
            printf("pmessage(%s, %d) = %d\n", msg, fatal, rc);

        XPUSHs(sv_2mortal(newSVuv(rc)));

    }
    XSRETURN(1);
} // pmessage


////////////////////////////////////////////////////////////////////////
//                         pget($refname)                             //
////////////////////////////////////////////////////////////////////////
XS(pget)
{
    dXSARGS;
    if (items != 1  &&  items != 2)
        croak("Usage: pget(refname [,GEO])");
    SP -= items;
    {
        char         *sym = (char *)SvPV_nolen(ST(0));
	Param        *p   = get_param(sym);
        u_int32_t    geo  = Param::GEO_DEF;

        // GEO
        if (items > 1)
            geo = SvUV(ST(1));

        string RETVAL = p->geti(geo);
        if (debugPerl)
        {
            if (geo != Param::GEO_DEF)
                printf("pget(%s, 0x%08x) = %s\n", sym, geo, RETVAL.c_str());
            else
                printf("pget(%s) = %s\n", sym, RETVAL.c_str());
        }
        XPUSHs(sv_2mortal(newSVpv(RETVAL.c_str(), RETVAL.length())));
    }
    XSRETURN(1);
} // pget


////////////////////////////////////////////////////////////////////////
//                      pget_geos($refname)                           //
////////////////////////////////////////////////////////////////////////
XS(pget_geos)
{
    dXSARGS;
    if (items != 1)
        croak("Usage: pget_geos(refname)");
    SP -= items;
    {
        char                          *sym = (char *)SvPV_nolen(ST(0));
        Param                         *p   = get_param(sym);

        // Check if parameter exists
        if (p == NULL)
            croak("Symbol (refname) \"%s\" doesn't exist", sym);

        if (debugPerl)
            printf("pget_geos(%s) = ", sym);
        AV *RETVAL = newAV();
        for (int geo_idx = 0; geo_idx < p->values.size(); geo_idx++)
        {
            av_push(RETVAL, newSVuv(p->values.key(geo_idx)));
            if (debugPerl)
                printf("0x%x ", p->values.key(geo_idx));
        }
        if (debugPerl)
            printf("\n");

        //XPUSHs(newRV_noinc((SV*)RETVAL));
        XPUSHs(sv_2mortal(newRV_noinc((SV*)RETVAL)));
        //XPUSHs(sv_2mortal((SV*)RETVAL));
    }
    XSRETURN(1);
} // pget_geos


////////////////////////////////////////////////////////////////////////
//                     pset($refname, $value)                         //
////////////////////////////////////////////////////////////////////////
XS(pset)
{
    dXSARGS;
    if (items != 2  &&  items != 3)
        croak("Usage: pset(refname, value [,GEO])");
    SP -= items;
    {
        u_int32_t                     geo = Param::GEO_DEF;
        char                          *sym = (char *)SvPV_nolen(ST(0));
        char                          *val = (char *)SvPV_nolen(ST(1));

        Param                         *p   = get_param(sym);

        // GEO
        if (items > 2)
            geo = SvUV(ST(2));

        // Check if parameter exists
        if (p == NULL)
            croak("-E- %s:%d: Symbol (refname) \"%s\" doesn't exist", g_source, g_line, sym);

        if (debugPerl)
        {
            if (geo != Param::GEO_DEF)
                printf("pset(%s, %s, 0x%08x)\n", sym, val, geo);
            else
                printf("pset(%s, %s)\n", sym, val);
        }
        if (!p->assign(val, g_source , g_line, geo))
            croak("-E- %s:%d: Parameter \"%s\" assigment error: %s", g_source, g_line, sym, p->err());
    }
    XSRETURN_EMPTY;
} // pset


////////////////////////////////////////////////////////////////////////
//                      pexists($refname)                             //
////////////////////////////////////////////////////////////////////////
XS(pexists)
{
    dXSARGS;
    if (items != 1)
        croak("Usage: pexists(refname)");
    SP -= items;
    {
        char         *sym = (char *)SvPV_nolen(ST(0));
        unsigned int RETVAL;

        Param        *p = get_param(sym);

        RETVAL = (p != NULL) ? 1 : 0;
        if (debugPerl)
            printf("pexists(%s) = %d\n", sym, RETVAL);
        XPUSHs(sv_2mortal(newSViv(RETVAL)));
    }
    XSRETURN(1);
} // pexists


map<string, Param*>::iterator par_it;

////////////////////////////////////////////////////////////////////////
//                         pfirstkey()                                //
////////////////////////////////////////////////////////////////////////
XS(pfirstkey)
{
    dXSARGS;
    if (items != 0)
        croak("Usage: pfirstkey()");

    par_it = plist->params.begin();
    if (par_it == plist->params.end())
    {
        if (debugPerl)
            printf("pfirstkey() = ()\n");
        XSRETURN_EMPTY;
    }

    const char *key = par_it->first.c_str();
    string     val = par_it->second->get();
    ++par_it;

    if (debugPerl)
        printf("pfirstkey() = (%s, %s)\n", key, val.c_str());
    XPUSHs(sv_2mortal(newSVpv(val.c_str(), val.length())));
    XPUSHs(sv_2mortal(newSVpv(key, strlen(key))));
    XSRETURN(2);
} // pfirstkey


////////////////////////////////////////////////////////////////////////
//                       pnextkey($last)                              //
////////////////////////////////////////////////////////////////////////
XS(pnextkey)
{
    dXSARGS;
    if (items != 1)
        croak("Usage: pnextkey(last_key)");
    SP -= items;
    
    // orenk : Switch to refname map once done with name map:
    if (par_it == plist->params.end())
    {
	par_it = plist->refparams.begin();
        if (debugPerl)
            printf("pnextkey(): switched to refparams list\n");
    }

    if (par_it == plist->refparams.end())
    {
        if (debugPerl)
            printf("pnextkey() = ()\n");
        XSRETURN_EMPTY;
    }


    char       *last = (char *)SvPV_nolen(ST(0));
    const char *key = par_it->first.c_str();
    string     val = par_it->second->get();
    ++par_it;

    if (debugPerl)
        printf("pnextkey(%s) = (%s, %s)\n", last, key, val.c_str());
    XPUSHs(sv_2mortal(newSVpv(val.c_str(), val.length())));
    XPUSHs(sv_2mortal(newSVpv(key, strlen(key))));
    XSRETURN(2);
} // pnextkey


////////////////////////////////////////////////////////////////////////
Perl::Perl()
{
    static char *sargs[] = { "MIC", "-e", "" };

    perl_interp = perl_alloc();
    perl_construct(perl_interp);
    perl_parse(perl_interp, NULL, 3, sargs, (char **)NULL);

    // Define our rotines for PERL
    newXS("pcalculateGUID", pcalculateGUID, "MIC");
    newXS("pget",           pget,      "MIC");
    newXS("pset",           pset,      "MIC");
    newXS("pmessage",       pmessage,  "MIC"); // For backward compatibility with old brd files
    newXS("fget",           pget,      "MIC"); // For backward compatibility with old brd files
    newXS("fset",           pset,      "MIC"); // For backward compatibility with old brd files
    newXS("pget_geos",      pget_geos, "MIC");
    newXS("padd",           padd,      "MIC");
    newXS("pdelete",        pdelete,   "MIC");
    newXS("pexists",        pexists,   "MIC");
    newXS("pfirstkey",      pfirstkey, "MIC");
    newXS("pnextkey",       pnextkey,  "MIC");

    debugPerl = getenv("MIC_PERL_DEBUG") ? true : false;
} // Perl::Perl

////////////////////////////////////////////////////////////////////////
Perl::~Perl()
{
    perl_destruct(perl_interp);
    perl_free(perl_interp);
} // Perl::~Perl



//
// HACK:Another win perl hack
//
#ifdef bool
#undef bool
#endif
////////////////////////////////////////////////////////////////////////
bool Perl::eval(const string& expr)
{
    eval_sv(newSVpv(expr.c_str(), expr.length()), G_EVAL | G_DISCARD | G_NOARGS | G_KEEPERR);
    if (SvTRUE(ERRSV)) {
        printf ("Perl ERROR - %s\n", SvPV_nolen(ERRSV));
	return false;
    }
    else
	return true;
} // Perl::eval


////////////////////////////////////////////////////////////////////////
void Perl::pinit(ParamList* pl)
{
    dSP;

    plist = pl;

    // Call "pinit" Perl routine without arguments
    PUSHMARK(SP);
    call_pv("pinit", G_DISCARD | G_NOARGS);
} // Perl::pinit


////////////////////////////////////////////////////////////////////////
void Perl::peval(ParamList* pl)
{
    dSP;

    plist = pl;

    // Call "peval" Perl routine without arguments
    PUSHMARK(SP);
    call_pv("peval", G_DISCARD | G_NOARGS);
} // Perl::peval


void Perl::usr_sub(const char* sub)
{
    dSP;

    PUSHMARK(SP);
    call_pv(sub, G_DISCARD | G_NOARGS);
}

void Perl::SetSource(const char* source) 
{
    g_source = source;
}

void Perl::SetLine  (int line)
{
    g_line = line;
}
